using System;
using System.Collections;
using System.Reflection;
using gov.va.med.vbecs.Common.Log;
using gov.va.med.vbecs.DAL.VistALink.Client;
using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

namespace gov.va.med.vbecs.DAL.VAL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Stas Antropov</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>1/22/2004</CreationDate>
	///	<Note>
	///		The Food and Drug Administration classifies this software as a medical device.  
	///		As such, it may not be changed in any way. Modifications to this software may result 
	///		in an adulterated medical device under 21CFR820, the use of which is considered to be 
	///		a violation of US Federal Statutes.  Acquiring and implementing this software through 
	///		the Freedom of information Act requires the implementor to assume total responsibility 
	///		for the software, and become a registered manufacturer of a medical device, 
	///		subject to FDA regulations.
	///	</Note>
	/// <summary>
	/// VBECS-specific UI-bound implementation of VistALink <see cref="RpcBroker"/> - the class facilitating
	/// VistALink communication and execution of remote M procedures. Used internally by the VistALink Access Layer classes.
	/// </summary>

	#endregion	
	
	internal class VbecsRpcBroker : RpcBroker
	{
		private const string VISTA_SERVER_ERROR_CODE_LOGON_FAILED_DUE_TO_TOO_MANY_ATTEMPTS = "183005";

		private string _vbecsUserDuz;
		private IVistALinkUIBinder _uiBinder;
        // Logger object
        readonly ILogger _logger = LogManager.Instance().LoggerLocator.GetLogger(MethodBase.GetCurrentMethod().DeclaringType);

		/// <summary>
		/// Constructor creating an instance of the class using supplied UI binder and user DUZ (if known beforehand).
		/// </summary>
		/// <param name="uiBinder">VistALink UI binder to use to communicate to UI.</param>
		/// <param name="vbecsUserDuz">VistA user DUZ to restrict connection for (if known, null means that any user can connect).</param>
		internal VbecsRpcBroker( IVistALinkUIBinder uiBinder, string vbecsUserDuz ) : base()
		{
			if( uiBinder == null )
				throw( new ArgumentNullException( "uiBinder" ) );

			_uiBinder = uiBinder;
			_vbecsUserDuz = ( vbecsUserDuz == null ) ? null : vbecsUserDuz.Trim();
		}

		/// <summary>
		/// Performs logon checking for a specific problems and notifying user in appropriate manner.
		/// </summary>
		/// <param name="vistaServer">VistA server handle in application configuration file.</param>
		/// <returns>True if logon succeds, false on failure.</returns>
		internal new bool Logon( ServerConnectionInfo vistaServer )
		{
			if( vistaServer == null )
				throw( new ArgumentNullException( "vistaServer" ) );

			try
			{
				return base.Logon( vistaServer );
			}
			catch( SecurityFaultException xcp )
			{
				if( xcp.FaultInfo.FaultCode != FaultCode.Server || xcp.FaultInfo.ErrorCode.Trim() != VISTA_SERVER_ERROR_CODE_LOGON_FAILED_DUE_TO_TOO_MANY_ATTEMPTS )
					throw;
				
				//  o   any logon attempts caused server lockout
				UINotifyAboutProblem( Common.StrRes.SysErrMsg.UC018.VistAServerLockedDueToTooManyLogonAttempts( xcp.FaultInfo.ErrorMessage ).ResString );
				this.Close();
				return false;
			}
		}

		/// <summary>
		/// Implementation of parent class' abstract method getting access and verify codes from user.
		/// </summary>
		/// <param name="accessCode">Access code entered by user.</param>
		/// <param name="verifyCode">Verify code entered by user.</param>
		/// <param name="requestVerifyCodeChangeFlag">Flag indicating whether user wants to change VistA verify code.</param>
		/// <returns>True if user provides required data and confirms entry. False if user elects to cancel VistA sign-on.</returns>
		protected override bool UIGetAccessAndVerifyCodes( out VistASecurityCode accessCode, out VistASecurityCode verifyCode, out bool requestVerifyCodeChangeFlag )
		{
			return _uiBinder.GetVistAAccessAndVerifyCodes( this.RemoteServerSetupAndIntroInfo, this.ServerConnectionInfo, out accessCode, out verifyCode, out requestVerifyCodeChangeFlag );
		}

		/// <summary>
		/// Implementation of parent class' abstract method presenting user with list 
		/// of available divisions and getting a selection.
		/// </summary>
		/// <param name="divisionList">List of divisions available for user to choose from.</param>
		/// <param name="userSelectedDivision">Division selected by user.</param>
		/// <returns>True if user provides required data and confirms entry. False if user elects to cancel VistA sign-on.</returns>
		protected override bool UISelectDivision( DivisionInfoCollection divisionList, out DivisionInfo userSelectedDivision )
		{
			return _uiBinder.GetVistADivisionSelection( this.RemoteServerSetupAndIntroInfo, this.ServerConnectionInfo, divisionList, out userSelectedDivision );
		}

		/// <summary>
		/// Implementation of parent class' abstract method getting new verify code from user 
		/// to change verify code in VistA (on user's or server's request). 
		/// </summary>
		/// <param name="oldVerifyCode">Old verify code entered by user.</param>
		/// <param name="newVerifyCode">New verify code entered by user.</param>
		/// <param name="newCheckVerifyCode">New verify code entered by user for verification (supplied twice to ensure codes do match).</param>
		/// <returns>True if user provides required data and confirms entry. False if user elects to cancel VistA sign-on.</returns>
		protected override bool UIChangeVerifyCode( out VistASecurityCode oldVerifyCode, out VistASecurityCode newVerifyCode, out VistASecurityCode newCheckVerifyCode )
		{
			return _uiBinder.GetVistANewVerifyCode( this.RemoteServerSetupAndIntroInfo, this.ServerConnectionInfo, out oldVerifyCode, out newVerifyCode, out newCheckVerifyCode );
		}

		/// <summary>
		/// Implementation of parent class' abstract method notifying user about problems with the VistALink.
		/// </summary>
		/// <param name="message">Message to display to user.</param>
		protected override void UINotifyAboutProblem( string message )
		{
			_uiBinder.NotifyAboutProblem( message );
		}

		/// <summary>
		/// Changes division selecting in VistA by logging off 
		/// from the server and logging back on using previously entered access and verify codes.
		/// </summary>
		internal void ChangeDivision()
		{
			if( !IsLoggedOn )
				return; 

			try
			{
				var server = ServerConnectionInfo;
                _logger.Debug("Change division: close current connection");
				Close();
                _logger.Debug("Change division: current connection closed, logon to server now");
				Logon( server );
                _logger.Debug("Change division: successfully logged on to server");
			}
			catch( VistALinkNetworkErrorException ex)
			{
                _logger.Warn(string.Format("Change division: suppressed exception: {0}", ex.Message));
				// Suppressing an exception. .NET FCL classes leave no other way.
			}
		}

		/// <summary>
		/// Performs check of supplied VBECS user DUZ allowing user to logon to the VBECS with 
		/// his/her own VistA credentials only. Notifies user if DUZs don't match.
		/// 
		/// The check is skipped if DUZ provided during construction is null. 
		/// </summary>
		/// <returns>True if VBECS user DUZ matches DUZ returned by VistA system or VBECS user DUZ is null.</returns>
		protected override bool PerformPostLogonUserDemographicsCheck()
		{
			// DUZ is unknown beforehand (using VistALink from MUCs) 
			if( _vbecsUserDuz == null )
				return true;

			if( _vbecsUserDuz == this.RemoteUserInfo.Duz.Trim() )
				return true;
			
			UINotifyAboutProblem( Common.StrRes.SysErrMsg.UC018.DuzMismatchDetected( this.RemoteUserInfo.Duz.Trim(), _vbecsUserDuz.Trim() ).ResString );

			return false;
		}
	}
}
